const STORAGE_KEYS = {
  thresholds: 'thresholds',
  reminderEnabled: 'reminderEnabled'
};

const DEFAULT_THRESHOLDS = {
  greenToYellow: 1,
  yellowToOrange: 4,
  orangeToRed: 12
};

const WEEKLY_ALARM = 'weeklyReminder';

chrome.runtime.onInstalled.addListener(async () => {
  await ensureDefaults();
  await syncReminderAlarm();
  chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true });
});

chrome.runtime.onStartup.addListener(async () => {
  await ensureDefaults();
  await syncReminderAlarm();
});

chrome.storage.onChanged.addListener(async (changes, area) => {
  if (area !== 'local') return;
  if (changes[STORAGE_KEYS.reminderEnabled]) {
    await syncReminderAlarm();
  }
});

chrome.alarms.onAlarm.addListener(async (alarm) => {
  if (alarm.name !== WEEKLY_ALARM) return;
  const { thresholds } = await chrome.storage.local.get(STORAGE_KEYS.thresholds);
  const effective = { ...DEFAULT_THRESHOLDS, ...(thresholds || {}) };
  const stats = await computeStats(effective);
  const shameScore = stats.total > 0 ? Math.round((stats.red / stats.total) * 100) : 0;

  const message = `Tu Shame Score es ${shameScore}% - Tienes ${stats.red} bookmarks rojos esperando`;

  await chrome.notifications.create({
    type: 'basic',
    iconUrl: 'icons/icon128.png',
    title: 'BookmarkBrain',
    message
  });
});

async function ensureDefaults() {
  const data = await chrome.storage.local.get([STORAGE_KEYS.thresholds, STORAGE_KEYS.reminderEnabled]);
  if (!data[STORAGE_KEYS.thresholds]) {
    await chrome.storage.local.set({ [STORAGE_KEYS.thresholds]: DEFAULT_THRESHOLDS });
  }
  if (typeof data[STORAGE_KEYS.reminderEnabled] !== 'boolean') {
    await chrome.storage.local.set({ [STORAGE_KEYS.reminderEnabled]: false });
  }
}

async function syncReminderAlarm() {
  const { reminderEnabled } = await chrome.storage.local.get(STORAGE_KEYS.reminderEnabled);
  if (reminderEnabled) {
    await chrome.alarms.create(WEEKLY_ALARM, {
      delayInMinutes: 1,
      periodInMinutes: 60 * 24 * 7
    });
  } else {
    await chrome.alarms.clear(WEEKLY_ALARM);
  }
}

async function computeStats(thresholds) {
  const tree = await chrome.bookmarks.getTree();
  const all = [];
  extractBookmarks(tree[0], '', all);
  const stats = { total: all.length, red: 0 };
  for (const bookmark of all) {
    if (getColor(bookmark.dateAdded, thresholds) === 'red') {
      stats.red += 1;
    }
  }
  return stats;
}

function extractBookmarks(node, folderName, acc) {
  if (node.url) {
    acc.push({
      id: node.id,
      title: node.title || 'Sin titulo',
      url: node.url,
      dateAdded: node.dateAdded,
      folder: folderName || 'Root'
    });
  }
  if (node.children) {
    const folder = node.title || folderName;
    node.children.forEach((child) => extractBookmarks(child, folder, acc));
  }
}

function getColor(dateAdded, thresholds) {
  const months = (Date.now() - dateAdded) / (1000 * 60 * 60 * 24 * 30);
  if (months < thresholds.greenToYellow) return 'green';
  if (months < thresholds.yellowToOrange) return 'yellow';
  if (months < thresholds.orangeToRed) return 'orange';
  return 'red';
}
