const STORAGE_KEYS = {
  thresholds: 'thresholds',
  viewMode: 'viewMode',
  reminderEnabled: 'reminderEnabled',
  theme: 'theme'
};

const DEFAULT_THRESHOLDS = {
  greenToYellow: 1,
  yellowToOrange: 4,
  orangeToRed: 12
};

// Constantes de tiempo
const ONE_MONTH = 1000 * 60 * 60 * 24 * 30;

// Estado
let allBookmarks = [];
let currentFilter = 'all';
let currentSort = 'oldest';
let currentFolder = 'all';
let thresholds = { ...DEFAULT_THRESHOLDS };
let viewMode = 'detailed';
const selectedIds = new Set();
let lastSelectedIndex = null;
let lastRenderedIds = [];
let theme = 'dark';
let toastTimeout = null;

// Al cargar el popup
document.addEventListener('DOMContentLoaded', init);

async function init() {
  await loadSettings();
  await loadBookmarks();
  renderStats();
  renderFolderFilter();
  renderBookmarks();
  updateExportCount();
  setupEventListeners();
}

async function loadSettings() {
  const data = await chrome.storage.local.get([
    STORAGE_KEYS.thresholds,
    STORAGE_KEYS.viewMode,
    STORAGE_KEYS.reminderEnabled,
    STORAGE_KEYS.theme
  ]);

  thresholds = normalizeThresholds({ ...DEFAULT_THRESHOLDS, ...(data[STORAGE_KEYS.thresholds] || {}) });
  viewMode = data[STORAGE_KEYS.viewMode] || 'detailed';
  const reminderEnabled = Boolean(data[STORAGE_KEYS.reminderEnabled]);
  theme = data[STORAGE_KEYS.theme] || 'dark';

  document.getElementById('threshold-green').value = thresholds.greenToYellow;
  document.getElementById('threshold-yellow').value = thresholds.yellowToOrange;
  document.getElementById('threshold-orange').value = thresholds.orangeToRed;
  document.getElementById('weekly-reminder').checked = reminderEnabled;
  document.getElementById('theme-toggle').checked = theme === 'light';

  setViewMode(viewMode, false);
  applyTheme(theme);
}

async function persistSettings() {
  await chrome.storage.local.set({
    [STORAGE_KEYS.thresholds]: thresholds,
    [STORAGE_KEYS.viewMode]: viewMode,
    [STORAGE_KEYS.theme]: theme
  });
}

// Cargar todos los bookmarks recursivamente
async function loadBookmarks() {
  const tree = await chrome.bookmarks.getTree();
  allBookmarks = [];
  extractBookmarks(tree[0], '');
}

function extractBookmarks(node, folderName) {
  if (node.url) {
    allBookmarks.push({
      id: node.id,
      title: node.title || 'Untitled',
      url: node.url,
      dateAdded: node.dateAdded,
      folder: folderName || 'Root'
    });
  }
  if (node.children) {
    const folder = node.title || folderName;
    node.children.forEach((child) => extractBookmarks(child, folder));
  }
}

// Calcular edad en meses
function getAgeInMonths(dateAdded) {
  return (Date.now() - dateAdded) / ONE_MONTH;
}

function getAgeInDays(dateAdded) {
  return Math.floor((Date.now() - dateAdded) / (1000 * 60 * 60 * 24));
}

// Obtener color segun edad
function getColor(dateAdded) {
  const months = getAgeInMonths(dateAdded);
  if (months < thresholds.greenToYellow) return 'green';
  if (months < thresholds.yellowToOrange) return 'yellow';
  if (months < thresholds.orangeToRed) return 'orange';
  return 'red';
}

// Texto de edad legible
function getAgeText(dateAdded) {
  const diff = Date.now() - dateAdded;
  const days = Math.floor(diff / (1000 * 60 * 60 * 24));

  if (days < 1) return 'today';
  if (days === 1) return 'yesterday';
  if (days < 7) return `${days} days ago`;
  if (days < 30) return `${Math.floor(days / 7)} weeks ago`;
  if (days < 365) return `${Math.floor(days / 30)} months ago`;
  const years = Math.floor(days / 365);
  return `${years} ${years === 1 ? 'year' : 'years'} ago`;
}

// Renderizar stats
function renderStats() {
  const stats = {
    total: allBookmarks.length,
    green: allBookmarks.filter((b) => getColor(b.dateAdded) === 'green').length,
    yellow: allBookmarks.filter((b) => getColor(b.dateAdded) === 'yellow').length,
    orange: allBookmarks.filter((b) => getColor(b.dateAdded) === 'orange').length,
    red: allBookmarks.filter((b) => getColor(b.dateAdded) === 'red').length
  };

  const shameScore = stats.total > 0 ? Math.round((stats.red / stats.total) * 100) : 0;

  document.getElementById('stat-total').textContent = stats.total;
  document.getElementById('stat-green').textContent = stats.green;
  document.getElementById('stat-yellow').textContent = stats.yellow;
  document.getElementById('stat-orange').textContent = stats.orange;
  document.getElementById('stat-red').textContent = stats.red;
  document.getElementById('shame-score').textContent = shameScore + '%';

  const shameEl = document.getElementById('shame-score');
  if (shameScore > 50) {
    shameEl.classList.add('shame-high');
  } else {
    shameEl.classList.remove('shame-high');
  }

  const celebration = document.getElementById('celebration');
  if (stats.total > 0 && stats.red === 0) {
    celebration.hidden = false;
  } else {
    celebration.hidden = true;
  }
}

// Renderizar dropdown de carpetas
function renderFolderFilter() {
  const folders = [...new Set(allBookmarks.map((b) => b.folder))].sort();
  const select = document.getElementById('folder-filter');
  select.innerHTML = '<option value="all">Todas las carpetas</option>';
  folders.forEach((folder) => {
    const option = document.createElement('option');
    option.value = folder;
    option.textContent = folder;
    select.appendChild(option);
  });
}

// Filtrar y ordenar bookmarks
function getFilteredBookmarks() {
  let filtered = [...allBookmarks];

  if (currentFilter !== 'all') {
    filtered = filtered.filter((b) => getColor(b.dateAdded) === currentFilter);
  }

  if (currentFolder !== 'all') {
    filtered = filtered.filter((b) => b.folder === currentFolder);
  }

  switch (currentSort) {
    case 'oldest':
      filtered.sort((a, b) => a.dateAdded - b.dateAdded);
      break;
    case 'newest':
      filtered.sort((a, b) => b.dateAdded - a.dateAdded);
      break;
    case 'alpha':
      filtered.sort((a, b) => a.title.localeCompare(b.title));
      break;
    case 'folder':
      filtered.sort((a, b) => a.folder.localeCompare(b.folder) || a.title.localeCompare(b.title));
      break;
  }

  return filtered;
}

function getSelectedBookmarks(filtered) {
  if (selectedIds.size === 0) return filtered;
  return filtered.filter((b) => selectedIds.has(b.id));
}

function updateExportCount() {
  const el = document.getElementById('export-count');
  const count = selectedIds.size;
  el.textContent = `${count} selected`;
}

function syncSelectionWithFilter(filtered) {
  const ids = new Set(filtered.map((b) => b.id));
  let changed = false;
  for (const id of selectedIds) {
    if (!ids.has(id)) {
      selectedIds.delete(id);
      changed = true;
    }
  }
  if (changed) {
    updateExportCount();
  }
}

// Renderizar lista de bookmarks
function renderBookmarks() {
  const container = document.getElementById('bookmarks-list');
  const filtered = getFilteredBookmarks();

  if (filtered.length === 0) {
    container.innerHTML =
      currentFilter === 'all' && currentFolder === 'all'
        ? '<div class="empty-state">No bookmarks saved yet</div>'
        : '<div class="empty-state">No results for this filter</div>';
    return;
  }

  syncSelectionWithFilter(filtered);
  container.classList.toggle('compact-mode', viewMode === 'compact');
  lastRenderedIds = filtered.map((b) => b.id);

  container.innerHTML = filtered
    .map((bookmark, index) => {
      const checked = selectedIds.has(bookmark.id) ? 'checked' : '';
      const selected = selectedIds.has(bookmark.id) ? 'selected' : '';
      if (viewMode === 'compact') {
        return `
          <div class="bookmark-item ${selected}" data-id="${bookmark.id}">
            <label class="select-box" title="Seleccionar">
              <input type="checkbox" class="select-bookmark" data-id="${bookmark.id}" data-index="${index}" ${checked} />
            </label>
            <span class="color-badge color-${getColor(bookmark.dateAdded)}"></span>
            <div class="bookmark-info">
              <a href="${bookmark.url}" class="bookmark-title" target="_blank" title="${bookmark.url}">
                ${escapeHtml(truncate(bookmark.title, 60))}
              </a>
            </div>
          </div>
        `;
      }

      return `
        <div class="bookmark-item ${selected}" data-id="${bookmark.id}">
          <label class="select-box" title="Seleccionar">
            <input type="checkbox" class="select-bookmark" data-id="${bookmark.id}" data-index="${index}" ${checked} />
          </label>
          <span class="color-badge color-${getColor(bookmark.dateAdded)}"></span>
          <div class="bookmark-info">
            <a href="${bookmark.url}" class="bookmark-title" target="_blank" title="${bookmark.url}">
              ${escapeHtml(truncate(bookmark.title, 50))}
            </a>
            <div class="bookmark-meta">
              <span class="bookmark-age">${getAgeText(bookmark.dateAdded)}</span>
              <span class="bookmark-folder">${escapeHtml(bookmark.folder)}</span>
            </div>
          </div>
          <div class="bookmark-actions">
            <button class="btn-open" data-url="${bookmark.url}" title="Open">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <path d="M18 13v6a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2V8a2 2 0 0 1 2-2h6"></path>
                <polyline points="15 3 21 3 21 9"></polyline>
                <line x1="10" y1="14" x2="21" y2="3"></line>
              </svg>
            </button>
            <button class="btn-delete" data-id="${bookmark.id}" title="Delete">
              <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                <line x1="18" y1="6" x2="6" y2="18"></line>
                <line x1="6" y1="6" x2="18" y2="18"></line>
              </svg>
            </button>
          </div>
        </div>
      `;
    })
    .join('');
}

// Utilidades
function truncate(str, len) {
  const safe = str || '';
  return safe.length > len ? safe.substring(0, len) + '...' : safe;
}

function escapeHtml(str) {
  const div = document.createElement('div');
  div.textContent = str || '';
  return div.innerHTML;
}

function setViewMode(mode, persist = true) {
  viewMode = mode;
  document.getElementById('view-detailed').classList.toggle('active', mode === 'detailed');
  document.getElementById('view-compact').classList.toggle('active', mode === 'compact');
  if (persist) {
    chrome.storage.local.set({ [STORAGE_KEYS.viewMode]: viewMode });
  }
  renderBookmarks();
}

// Export
function exportBookmarks(format) {
  const filtered = getFilteredBookmarks();
  const target = getSelectedBookmarks(filtered);
  if (target.length === 0) {
    alert('No bookmarks selected for export.');
    return;
  }
  exportDataset(target, format);
  showToast(`Exported ${target.length} bookmarks`);
}

function exportFiltered(format) {
  const filtered = getFilteredBookmarks();
  if (filtered.length === 0) {
    alert('No filtered bookmarks to export.');
    return;
  }
  exportDataset(filtered, format);
  showToast(`Exported ${filtered.length} bookmarks`);
}

function exportDataset(dataset, format) {
  const rows = dataset.map((b) => {
    const ageDays = getAgeInDays(b.dateAdded);
    return {
      titulo: b.title,
      url: b.url,
      carpeta: b.folder,
      edad_dias: ageDays,
      color: getColor(b.dateAdded),
      fecha_guardado: new Date(b.dateAdded).toISOString()
    };
  });

  const date = new Date();
  const yyyy = date.getFullYear();
  const mm = String(date.getMonth() + 1).padStart(2, '0');
  const dd = String(date.getDate()).padStart(2, '0');

  if (format === 'json') {
    const blob = new Blob([JSON.stringify(rows, null, 2)], { type: 'application/json' });
    triggerDownload(blob, `bookmarks-export-${yyyy}-${mm}-${dd}.json`);
    return;
  }

  const header = ['titulo', 'url', 'carpeta', 'edad_dias', 'color', 'fecha_guardado'];
  const csvRows = [header.join(',')];
  rows.forEach((row) => {
    const values = header.map((key) => csvEscape(String(row[key] ?? '')));
    csvRows.push(values.join(','));
  });
  const blob = new Blob([csvRows.join('\n')], { type: 'text/csv' });
  triggerDownload(blob, `bookmarks-export-${yyyy}-${mm}-${dd}.csv`);
}

function csvEscape(value) {
  if (value.includes('"') || value.includes(',') || value.includes('\n')) {
    return `"${value.replace(/"/g, '""')}"`;
  }
  return value;
}

function triggerDownload(blob, filename) {
  const url = URL.createObjectURL(blob);
  const link = document.createElement('a');
  link.href = url;
  link.download = filename;
  document.body.appendChild(link);
  link.click();
  link.remove();
  URL.revokeObjectURL(url);
}

function showToast(message) {
  let toast = document.getElementById('toast');
  if (!toast) {
    toast = document.createElement('div');
    toast.id = 'toast';
    toast.className = 'toast';
    document.body.appendChild(toast);
  }
  toast.textContent = message;
  toast.classList.add('show');
  if (toastTimeout) {
    clearTimeout(toastTimeout);
  }
  toastTimeout = setTimeout(() => {
    toast.classList.remove('show');
  }, 2000);
}

function applyTheme(nextTheme) {
  theme = nextTheme;
  document.body.classList.toggle('theme-light', theme === 'light');
}

function normalizeThresholds(values) {
  const next = { ...values };
  next.greenToYellow = Math.max(0, Number(next.greenToYellow) || DEFAULT_THRESHOLDS.greenToYellow);
  next.yellowToOrange = Math.max(next.greenToYellow, Number(next.yellowToOrange) || DEFAULT_THRESHOLDS.yellowToOrange);
  next.orangeToRed = Math.max(next.yellowToOrange, Number(next.orangeToRed) || DEFAULT_THRESHOLDS.orangeToRed);
  return next;
}

// Event listeners
function setupEventListeners() {
  document.querySelectorAll('.filter-btn').forEach((btn) => {
    btn.addEventListener('click', (e) => {
      document.querySelectorAll('.filter-btn').forEach((b) => b.classList.remove('active'));
      e.target.classList.add('active');
      currentFilter = e.target.dataset.filter;
      renderBookmarks();
    });
  });

  document.getElementById('folder-filter').addEventListener('change', (e) => {
    currentFolder = e.target.value;
    renderBookmarks();
  });

  document.getElementById('sort-select').addEventListener('change', (e) => {
    currentSort = e.target.value;
    renderBookmarks();
  });

  document.getElementById('view-detailed').addEventListener('click', () => {
    setViewMode('detailed');
  });

  document.getElementById('view-compact').addEventListener('click', () => {
    setViewMode('compact');
  });

  document.getElementById('export-btn').addEventListener('click', () => {
    const format = document.getElementById('export-format').value;
    exportBookmarks(format);
  });

  document.getElementById('export-filtered-btn').addEventListener('click', () => {
    const format = document.getElementById('export-format').value;
    exportFiltered(format);
  });

  document.getElementById('export-select-all').addEventListener('click', () => {
    const filtered = getFilteredBookmarks();
    filtered.forEach((b) => selectedIds.add(b.id));
    renderBookmarks();
    updateExportCount();
  });

  document.getElementById('export-clear').addEventListener('click', () => {
    selectedIds.clear();
    renderBookmarks();
    updateExportCount();
  });

  document.getElementById('threshold-green').addEventListener('input', async (e) => {
    thresholds.greenToYellow = Number(e.target.value) || DEFAULT_THRESHOLDS.greenToYellow;
    thresholds = normalizeThresholds(thresholds);
    document.getElementById('threshold-green').value = thresholds.greenToYellow;
    document.getElementById('threshold-yellow').value = thresholds.yellowToOrange;
    document.getElementById('threshold-orange').value = thresholds.orangeToRed;
    await persistSettings();
    renderStats();
    renderBookmarks();
  });

  document.getElementById('threshold-yellow').addEventListener('input', async (e) => {
    thresholds.yellowToOrange = Number(e.target.value) || DEFAULT_THRESHOLDS.yellowToOrange;
    thresholds = normalizeThresholds(thresholds);
    document.getElementById('threshold-green').value = thresholds.greenToYellow;
    document.getElementById('threshold-yellow').value = thresholds.yellowToOrange;
    document.getElementById('threshold-orange').value = thresholds.orangeToRed;
    await persistSettings();
    renderStats();
    renderBookmarks();
  });

  document.getElementById('threshold-orange').addEventListener('input', async (e) => {
    thresholds.orangeToRed = Number(e.target.value) || DEFAULT_THRESHOLDS.orangeToRed;
    thresholds = normalizeThresholds(thresholds);
    document.getElementById('threshold-green').value = thresholds.greenToYellow;
    document.getElementById('threshold-yellow').value = thresholds.yellowToOrange;
    document.getElementById('threshold-orange').value = thresholds.orangeToRed;
    await persistSettings();
    renderStats();
    renderBookmarks();
  });

  document.getElementById('weekly-reminder').addEventListener('change', async (e) => {
    await chrome.storage.local.set({ [STORAGE_KEYS.reminderEnabled]: e.target.checked });
  });

  document.getElementById('theme-toggle').addEventListener('change', async (e) => {
    applyTheme(e.target.checked ? 'light' : 'dark');
    await persistSettings();
  });

  document.getElementById('bookmarks-list').addEventListener('click', async (e) => {
    const deleteBtn = e.target.closest('.btn-delete');
    const openBtn = e.target.closest('.btn-open');
    const selectBox = e.target.closest('.select-bookmark');

    if (selectBox) {
      const id = selectBox.dataset.id;
      const index = Number(selectBox.dataset.index);
      if (e.shiftKey && lastSelectedIndex !== null && !Number.isNaN(index)) {
        const [start, end] = index < lastSelectedIndex ? [index, lastSelectedIndex] : [lastSelectedIndex, index];
        for (let i = start; i <= end; i += 1) {
          const rangeId = lastRenderedIds[i];
          if (rangeId) {
            selectedIds.add(rangeId);
          }
        }
        renderBookmarks();
      } else {
        if (selectBox.checked) {
          selectedIds.add(id);
        } else {
          selectedIds.delete(id);
        }
      }
      lastSelectedIndex = Number.isNaN(index) ? lastSelectedIndex : index;
      updateExportCount();
      return;
    }

    if (deleteBtn) {
      const id = deleteBtn.dataset.id;
      if (confirm('Delete this bookmark?')) {
        await chrome.bookmarks.remove(id);
        allBookmarks = allBookmarks.filter((b) => b.id !== id);
        selectedIds.delete(id);
        renderStats();
        renderFolderFilter();
        renderBookmarks();
        updateExportCount();
      }
    }

    if (openBtn) {
      const url = openBtn.dataset.url;
      if (url) {
        window.open(url, '_blank');
      }
    }
  });

  document.getElementById('delete-all-red').addEventListener('click', async () => {
    const redBookmarks = allBookmarks.filter((b) => getColor(b.dateAdded) === 'red');
    if (redBookmarks.length === 0) {
      alert('No red bookmarks to delete');
      return;
    }

    if (confirm(`Delete ${redBookmarks.length} red bookmarks? This action cannot be undone.`)) {
      for (const bookmark of redBookmarks) {
        await chrome.bookmarks.remove(bookmark.id);
      }
      allBookmarks = allBookmarks.filter((b) => getColor(b.dateAdded) !== 'red');
      redBookmarks.forEach((b) => selectedIds.delete(b.id));
      renderStats();
      renderFolderFilter();
      renderBookmarks();
      updateExportCount();
    }
  });
}
